<!DOCTYPE html>
<html lang="id">
<head>
  <meta charset="UTF-8">
  <title>Notifications Demo</title>
  <meta name="viewport" content="width=device-width, initial-scale=1.0">

  <style>
    body {
      margin: 0;
      font-family: Arial, sans-serif;
      background: #f3f4f6;
    }

    .topbar {
      height: 60px;
      background: white;
      border-bottom: 1px solid #e5e7eb;
      display: flex;
      align-items: center;
      justify-content: flex-end;
      padding: 0 20px;
      position: relative;
    }

    .bell-btn {
      position: relative;
      border: none;
      background: none;
      cursor: pointer;
      padding: 8px;
      border-radius: 8px;
    }

    .bell-btn:hover {
      background: #f3f4f6;
    }

    .badge {
      position: absolute;
      top: 2px;
      right: 2px;
      background: #dc2626;
      color: white;
      font-size: 11px;
      width: 18px;
      height: 18px;
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
    }

    .panel {
      position: absolute;
      right: 20px;
      top: 70px;
      width: 360px;
      background: white;
      border-radius: 10px;
      box-shadow: 0 10px 30px rgba(0,0,0,0.1);
      border: 1px solid #e5e7eb;
      overflow: hidden;
      display: none;
      flex-direction: column;
      max-height: 500px;
    }

    .panel-header {
      padding: 14px;
      border-bottom: 1px solid #e5e7eb;
      display: flex;
      justify-content: space-between;
      align-items: center;
    }

    .panel-header h3 {
      margin: 0;
      font-size: 16px;
    }

    .panel-header p {
      margin: 2px 0 0;
      font-size: 12px;
      color: gray;
    }

    .mark-all {
      font-size: 12px;
      color: #2563eb;
      cursor: pointer;
      border: none;
      background: none;
    }

    .notif-list {
      overflow-y: auto;
    }

    .notif-item {
      padding: 12px 14px;
      border-bottom: 1px solid #f3f4f6;
      cursor: pointer;
      display: flex;
      gap: 10px;
    }

    .notif-item.unread {
      background: #eff6ff;
    }

    .notif-item:hover {
      background: #f9fafb;
    }

    .notif-icon {
      width: 32px;
      height: 32px;
      border-radius: 50%;
      background: #dcfce7;
      display: flex;
      align-items: center;
      justify-content: center;
      font-size: 14px;
    }

    .notif-content {
      flex: 1;
    }

    .notif-message {
      font-size: 13px;
      margin-bottom: 4px;
    }

    .notif-time {
      font-size: 11px;
      color: gray;
    }

    .close-btn {
      border: none;
      background: none;
      cursor: pointer;
      color: gray;
    }

    .empty {
      padding: 40px;
      text-align: center;
      color: gray;
      font-size: 13px;
    }

    .toast {
      position: fixed;
      top: 20px;
      right: 20px;
      background: #16a34a;
      color: white;
      padding: 14px 18px;
      border-radius: 8px;
      box-shadow: 0 10px 20px rgba(0,0,0,0.15);
      font-size: 14px;
      animation: slideIn 0.3s ease;
    }

    @keyframes slideIn {
      from { transform: translateX(20px); opacity: 0; }
      to { transform: translateX(0); opacity: 1; }
    }

    .demo-btn {
      margin: 20px;
      padding: 10px 14px;
      background: #dc2626;
      color: white;
      border: none;
      border-radius: 6px;
      cursor: pointer;
    }
  </style>
</head>

<body>

<div class="topbar">
  <button class="bell-btn" onclick="togglePanel()">
    🔔
    <div id="badge" class="badge" style="display:none;">0</div>
  </button>

  <div id="panel" class="panel">
    <div class="panel-header">
      <div>
        <h3>Notifikasi</h3>
        <p><span id="unreadText">0</span> belum dibaca</p>
      </div>
      <button class="mark-all" onclick="markAllAsRead()">Tandai semua</button>
    </div>

    <div id="notifList" class="notif-list"></div>
  </div>
</div>

<button class="demo-btn" onclick="simulateOrder()">
  Simulasi Pesanan Baru
</button>

<script>
  let notifications = [];
  let showPanel = false;

  function togglePanel() {
    showPanel = !showPanel;
    document.getElementById("panel").style.display = showPanel ? "flex" : "none";
  }

  function updateUI() {
    const badge = document.getElementById("badge");
    const list = document.getElementById("notifList");
    const unreadCount = notifications.filter(n => !n.read).length;

    document.getElementById("unreadText").innerText = unreadCount;

    if (unreadCount > 0) {
      badge.style.display = "flex";
      badge.innerText = unreadCount > 9 ? "9+" : unreadCount;
    } else {
      badge.style.display = "none";
    }

    if (notifications.length === 0) {
      list.innerHTML = '<div class="empty">Tidak ada notifikasi</div>';
      return;
    }

    list.innerHTML = "";

    notifications.forEach(notif => {
      const div = document.createElement("div");
      div.className = "notif-item " + (!notif.read ? "unread" : "");
      div.onclick = () => markAsRead(notif.id);

      div.innerHTML = `
        <div class="notif-icon">${notif.type === 'new_order' ? '🕒' : '✅'}</div>
        <div class="notif-content">
          <div class="notif-message">${notif.message}</div>
          <div class="notif-time">${notif.time}</div>
        </div>
        <button class="close-btn" onclick="event.stopPropagation(); clearNotif('${notif.id}')">✕</button>
      `;

      list.appendChild(div);
    });
  }

  function showToast(message) {
    const toast = document.createElement("div");
    toast.className = "toast";
    toast.innerText = message;
    document.body.appendChild(toast);

    setTimeout(() => toast.remove(), 3000);
  }

  function simulateOrder() {
    const orderNumber = Math.floor(Math.random() * 1000);
    const notif = {
      id: "notif-" + Date.now(),
      message: "Pesanan baru: ORD-" + orderNumber,
      type: "new_order",
      read: false,
      time: new Date().toLocaleTimeString("id-ID", {
        hour: '2-digit',
        minute: '2-digit'
      })
    };

    notifications.unshift(notif);
    showToast(notif.message);
    updateUI();
  }

  function markAsRead(id) {
    notifications = notifications.map(n =>
      n.id === id ? { ...n, read: true } : n
    );
    updateUI();
  }

  function markAllAsRead() {
    notifications = notifications.map(n => ({ ...n, read: true }));
    updateUI();
  }

  function clearNotif(id) {
    notifications = notifications.filter(n => n.id !== id);
    updateUI();
  }

  updateUI();
</script>

</body>
</html>